import AIAlignment.PenaltyERT

/-!
# PenaltyERT_MR

Adds an abstract motivated-reasoning (MR) penalty term to the ERT core.

We keep MR abstract (a function into ℝ) and prove structural properties *under explicit
assumptions*.

This is epistemically responsible: it avoids claiming boundedness unless MR is assumed bounded.
-/

namespace AIAlignment

noncomputable section

/-- Full ERT penalty: calibration loss plus β * MR. -/
def Lert (k x hatT β : ℝ) (MR : ℝ) : ℝ := Lcal k x hatT + β * MR

lemma Lert_nonneg (k x hatT β MR : ℝ)
    (hβ : 0 ≤ β) (hMR : 0 ≤ MR) : 0 ≤ Lert k x hatT β MR := by
  dsimp [Lert]
  have hcal : 0 ≤ Lcal k x hatT := Lcal_nonneg k x hatT
  nlinarith

/--
If the calibration loss is bounded by 1 (given k>0, x≥0, hatT∈[0,1]) and MR is bounded by M,
then the full ERT penalty is bounded by 1 + β*M.
-/
lemma Lert_le (k x hatT β MR M : ℝ)
    (hk : 0 < k) (hx : 0 ≤ x)
    (hhat0 : 0 ≤ hatT) (hhat1 : hatT ≤ 1)
    (hβ : 0 ≤ β)
    (hMR : MR ≤ M) : Lert k x hatT β MR ≤ 1 + β * M := by
  dsimp [Lert]
  have hcal : Lcal k x hatT ≤ 1 := Lcal_le_one k x hatT hk hx hhat0 hhat1
  have hβMR : β * MR ≤ β * M := mul_le_mul_of_nonneg_left hMR hβ
  nlinarith

end

end AIAlignment
